<?php

declare(strict_types=1);

namespace Erlage\Photogram;

use R;
use Exception;
use Erlage\Photogram\Data\Tables\Sys\RestTable;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Sys\ResponseTable;

final class Response
{
    /**
     * @return Response 
     */
    final public static function instance()
    {
        static $instance = false;

        if (false === $instance)
        {
            $instance = new Response();
        }

        return $instance;
    }

    /**
     * @var array
     */
    protected $response;

    public function init(): self
    {
        $this -> response = array(
            RestTable::REQUEST => array(
                RequestTable::REQ_TYPE    => State::getReqType(),
                RequestTable::API_VERSION => State::getApiVersion(),
            ),

            RestTable::RESPONSE => array(
                ResponseTable::MESSAGE => '',
                ResponseTable::CONTENT => array('ready' => 'ok'),
            ),
        );

        return $this;
    }

    public function getResponseData(): array
    {
        return $this -> response;
    }

    public function setMessage(string $message): self
    {
        $this -> response[RestTable::RESPONSE][ResponseTable::MESSAGE] = $message;

        return $this;
    }

    public function setContent(string $namespace, array $content): self
    {
        if ('' !== $namespace)
        {
            $this -> response[RestTable::RESPONSE][ResponseTable::CONTENT][$namespace] = $content;
        }
        else
        {
            $this -> response[RestTable::RESPONSE][ResponseTable::CONTENT] = $content;
        }

        return $this;
    }

    public function setAdditionalContent(string $namespace, array $content): self
    {
        $this
            -> response[RestTable::RESPONSE]
                [ResponseTable::CONTENT]
                [RestTable::ADDITIONAL_OBJECTS]
                [$namespace] = $content;

        return $this;
    }

    public function dispatch(): void
    {
        if (State::isDispatched())
        {
            throw new Exception('Already dispatched');
        }

        \header('Content-Type: application/json; charset=utf-8');

        echo \json_encode($this -> response);

        State::setDispatched(true);

        if (MODE_LOG_RESPONSE_DISPATCH)
        {
            SystemLogger::response(
                array(
                    'Gathered request data'   => Request::instance() -> getRequestData(),
                    'Actual response payload' => $this -> getResponseData(),
                )
            );
        }

        if (MODE_LOG_DATABASE_ADAPTER)
        {
            SystemLogger::databaseAdapter(
                array(
                    'Gathered request data'   => Request::instance() -> getRequestData(),
                    'Database Adapter logs'   => R::getLogs(),
                )
            );
        }

        exit();
    }
}
